<?php
/**
 * Subject Details Page
 * 
 * Displays detailed information about a specific subject including
 * university, lecturer, pricing, and subscription information.
 * 
 * @author Thamrah Development Team
 * @version 2.0
 * @since 2025-06-14
 */

// Start session with secure settings
ini_set('session.cookie_httponly', 1);
ini_set('session.cookie_secure', 1);
ini_set('session.use_strict_mode', 1);
session_start();

// Set content type and security headers
header('Content-Type: text/html; charset=UTF-8');
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');

require_once("db/connection.php");

/**
 * Initialize user and locale variables with proper validation
 */
function initializeUserSession() {
    $data = [
        'id' => 0,
        'email' => '',
        'user' => null,
        'locale' => 'ar'
    ];
    
    // Get locale from various sources with fallback
    $accept_language = $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? 'ar';
    $data['locale'] = substr($accept_language, 0, 2);
    
    // Override with session language if set
    if (isset($_SESSION["lang"]) && in_array($_SESSION["lang"], ['en', 'ar'])) {
        $data['locale'] = $_SESSION["lang"];
    }
    
    // Get user email from session
    if (isset($_SESSION["email"]) && !empty($_SESSION["email"])) {
        $data['email'] = sanitizeInput($_SESSION["email"]);
    }
    
    return $data;
}

/**
 * Get subject data with proper validation
 * 
 * @param mysqli $connection Database connection
 * @param int $subject_id Subject ID
 * @param string $locale Language locale
 * @return array Subject data or error
 */
function getSubjectData($connection, $subject_id, $locale) {
    try {
        // Get subject basic information
        $query = "SELECT s.*, st.name, st.description 
                  FROM subjects s 
                  LEFT JOIN subject_translations st ON s.id = st.subject_id 
                  WHERE s.id = ? AND s.status = 'active' AND st.locale = ?";
        
        $result = executeQuery($connection, $query, [$subject_id, $locale], 'is');
        
        if ($result === false || $result->num_rows === 0) {
            return ['error' => 'Subject not found'];
        }
        
        return $result->fetch_assoc();
        
    } catch (Exception $e) {
        error_log("Error fetching subject data: " . $e->getMessage());
        return ['error' => 'Database error'];
    }
}

/**
 * Get user data by email
 * 
 * @param mysqli $connection Database connection
 * @param string $email User email
 * @return array|null User data or null if not found
 */
function getUserData($connection, $email) {
    if (empty($email)) {
        return null;
    }
    
    try {
        $query = "SELECT * FROM users WHERE email = ? LIMIT 1";
        $result = executeQuery($connection, $query, [$email], 's');
        
        if ($result === false || $result->num_rows === 0) {
            return null;
        }
        
        return $result->fetch_assoc();
        
    } catch (Exception $e) {
        error_log("Error fetching user data: " . $e->getMessage());
        return null;
    }
}

/**
 * Get settings data with translations
 * 
 * @param mysqli $connection Database connection
 * @param string $locale Language locale
 * @return array Settings data
 */
function getSettingsData($connection, $locale) {
    try {
        $query = "SELECT s.*, st.* 
                  FROM settings s 
                  LEFT JOIN setting_translations st ON s.id = st.setting_id 
                  WHERE st.locale = ? 
                  LIMIT 1";
        
        $result = executeQuery($connection, $query, [$locale], 's');
        
        if ($result === false || $result->num_rows === 0) {
            return [];
        }
        
        return $result->fetch_assoc();
        
    } catch (Exception $e) {
        error_log("Error fetching settings: " . $e->getMessage());
        return [];
    }
}

// Main execution
try {
    // Check database connection
    if (!$con) {
        die("Database connection failed. Please try again later.");
    }
    
    // Initialize session data
    $session_data = initializeUserSession();
    extract($session_data);
    
    // Validate and get subject ID
    $subject_id = validateInteger($_GET['id'] ?? 0);
    if ($subject_id === false || $subject_id <= 0) {
        header("Location: index.php");
        exit();
    }
    
    // Get user data if email is set
    if (!empty($email)) {
        $user = getUserData($con, $email);
        if ($user) {
            $id = (int)$user['id'];
        }
    }
    
    // Get subject data
    $subject_data = getSubjectData($con, $subject_id, $locale);
    if (isset($subject_data['error'])) {
        header("Location: index.php");
        exit();
    }
    
    // Get settings
    $settings_data = getSettingsData($con, $locale);
    
    // Set page variables
    $title = ($subject_data['name'] ?? 'Subject') . " - Thamrah";
    $dir = ($locale == "en") ? "ltr" : "rtl";
    $lang = ($locale == "en") ? "ar" : "en";
    $style = $locale == "ar" ? "rtl-style" : "";
    
    // Include language files safely
    $website_lang_file = "lang/{$locale}/website.php";
    $cp_lang_file = "lang/{$locale}/cp.php";
    
    if (file_exists($website_lang_file)) {
        require_once($website_lang_file);
    }
    if (file_exists($cp_lang_file)) {
        require_once($cp_lang_file);
    }
    
    // Ensure $website array exists
    if (!isset($website)) {
        $website = [];
    }
    
} catch (Exception $e) {
    error_log("Error in subject_details.php: " . $e->getMessage());
    header("Location: index.php");
    exit();
}
?>

<!DOCTYPE html>
<html lang="<?php echo htmlspecialchars($locale); ?>" dir="<?php echo htmlspecialchars($dir); ?>">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=0">
    <meta name="description" content="<?php echo htmlspecialchars($subject_data['description'] ?? ''); ?>">
    <title><?php echo htmlspecialchars($title); ?></title>

    <!-- Stylesheets -->
    <link rel="icon" href="images/favicon.svg">
    <link href="/css/bootstrap.css" rel="stylesheet">
    <link href="/css/style.css" rel="stylesheet">
    <link href="/css/responsive.css" rel="stylesheet">
  
    <!-- Scripts -->
    <script src="/js/jquery-3.2.1.min.js"></script>
    <script src='https://www.google.com/recaptcha/api.js' async defer></script>
    <script src="https://player.vimeo.com/api/player.js"></script>
</head>

<body>
    <div class="main-wrapper pd-page">
        <!-- Header Section -->
        <header id="header" class="header-page">
            <div class="container">
                <div class="logo-site">
                    <a href="/index.php">
                        <img src="/images/logo.svg" alt="Thamrah Logo" />
                    </a>
                </div>
                <div class="menu-end">
                    <div class="lang-st">
                        <a href="change_language.php?lang=<?php echo htmlspecialchars($lang); ?>&page=<?php echo urlencode($_SERVER['REQUEST_URI']); ?>">
                            <i class="icon icon-languge"></i> 
                            <?php echo htmlspecialchars($website['lang'] ?? 'Language'); ?>
                        </a>
                    </div>
                    <button type="button" class="hamburger" aria-label="Toggle menu">
                        <span class="hamb-top"></span>
                        <span class="hamb-middle"></span>
                        <span class="hamb-bottom"></span>
                    </button>
                </div>
                
                <!-- Navigation Menu -->
                <div class="side-menu">
                    <div class="close-menu">
                        <i class="fa-solid fa-xmark"></i>
                    </div>
                    <ul class="main_menu clearfix">
                        <li><a class="page-scroll" href="/index.php"><?php echo htmlspecialchars($website['Home'] ?? 'Home'); ?></a></li>
                        <li><a class="page-scroll" href="/about.php"><?php echo htmlspecialchars($website['About Us'] ?? 'About Us'); ?></a></li>
                        <li><a class="page-scroll" href="/faq.php"><?php echo htmlspecialchars($website['Faqs'] ?? 'FAQs'); ?></a></li>
                        <li><a class="page-scroll" href="/contact.php"><?php echo htmlspecialchars($website['Contact Us'] ?? 'Contact Us'); ?></a></li>
                        
                        <?php if ($id === 0): ?>
                            <li>
                                <a style="background: #fbaf41; padding: 5px 15px; border-radius: 5px; color: #000;" href="/login.php">
                                    <span><?php echo htmlspecialchars($website["login"] ?? 'Login'); ?></span>
                                </a>
                            </li>
                        <?php else: ?>
                            <li>
                                <a style="background: #fbaf41; padding: 5px 15px; border-radius: 5px; color: #000; font-weight: bolder; font-size: 14px;" href="/mysubjects.php">
                                    <span><?php echo htmlspecialchars($website["mySubjects"] ?? 'My Subjects'); ?></span>
                                </a>
                            </li>
                            <li>
                                <a style="background: #fbaf41; padding: 5px 15px; border-radius: 5px; color: #000; font-weight: bolder; font-size: 14px;" href="/myaccount.php">
                                    <span><?php echo htmlspecialchars($website["my_account"] ?? 'My Account'); ?></span>
                                </a>
                            </li>
                            <li>
                                <a style="background: #fbaf41; padding: 5px 15px; border-radius: 5px; color: #000; font-weight: bolder; font-size: 14px;" href="/logout.php">
                                    <span><?php echo htmlspecialchars($website["logout"] ?? 'Logout'); ?></span>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>

                    <!-- Social Media Links -->
                    <?php if (!empty($settings_data)): ?>
                    <div class="social-media">
                        <ul>
                            <?php if (!empty($settings_data['instagram'])): ?>
                                <li><a href="<?php echo htmlspecialchars($settings_data['instagram']); ?>" target="_blank" rel="noopener"><i class="icon icon-instagram"></i></a></li>
                            <?php endif; ?>
                            <?php if (!empty($settings_data['twitter'])): ?>
                                <li><a href="<?php echo htmlspecialchars($settings_data['twitter']); ?>" target="_blank" rel="noopener"><i class="fa-brands fa-x-twitter"></i></a></li>
                            <?php endif; ?>
                            <?php if (!empty($settings_data['facebook'])): ?>
                                <li><a href="<?php echo htmlspecialchars($settings_data['facebook']); ?>" target="_blank" rel="noopener"><i class="icon icon-facebook"></i></a></li>
                            <?php endif; ?>
                            <?php if (!empty($settings_data['youtube'])): ?>
                                <li><a href="<?php echo htmlspecialchars($settings_data['youtube']); ?>" target="_blank" rel="noopener"><i class="icon icon-youtube"></i></a></li>
                            <?php endif; ?>
                        </ul>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </header>

        <!-- Main Content -->
        <main>
            <!-- Breadcrumb -->
            <div class="breadcrumb-bar">
                <div class="container">
                    <ul class="breadcrumb">
                        <li><a href="index.php"><?php echo htmlspecialchars($website['Home'] ?? 'Home'); ?></a></li>
                        <li><?php echo htmlspecialchars($website['Subject'] ?? 'Subject'); ?></li>
                    </ul>
                </div>
            </div>

            <!-- Subject Details Section -->
            <section class="subject_details_page">
                <div class="content--subject">
                    <div class="container">
                        <div class="row">
                            <div class="col-lg-8">
                                <div class="dta--subject wow fadeInUp">
                                    <h1><?php echo htmlspecialchars($subject_data['name'] ?? 'Subject'); ?></h1>
                                    
                                    <?php if (!empty($subject_data['description'])): ?>
                                        <div class="subject-description">
                                            <p><?php echo htmlspecialchars($subject_data['description']); ?></p>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <!-- Subject Statistics -->
                                    <div class="num-subject">
                                        <div>
                                            <figure><i class="fa-solid fa-eye"></i></figure>
                                            <p><?php echo number_format($subject_data['num_views'] ?? 0); ?></p>
                                        </div>
                                        <div>
                                            <figure><i class="icon icon-chapter"></i></figure>
                                            <p><?php echo htmlspecialchars($website['Chapters'] ?? 'Chapters'); ?> <?php echo number_format($subject_data['num_chapters'] ?? 0); ?></p>
                                        </div>
                                        <div>
                                            <figure><i class="icon icon-lesson"></i></figure>
                                            <p><?php echo htmlspecialchars($website['Lessons'] ?? 'Lessons'); ?> <?php echo number_format($subject_data['num_lessons'] ?? 0); ?></p>
                                        </div>
                                        <div>
                                            <figure><i class="fa-solid fa-clock"></i></figure>
                                            <p><?php echo htmlspecialchars($website['total hours'] ?? 'Total Hours'); ?> <?php echo number_format($subject_data['total_hours'] ?? 0); ?></p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-lg-4">
                                <!-- Subject Image/Video -->
                                <div class="subject-media">
                                    <?php 
                                    $subject_image = !empty($subject_data['image']) ? $subject_data['image'] : 'images/default.png';
                                    ?>
                                    <img src="<?php echo htmlspecialchars($subject_image); ?>" alt="<?php echo htmlspecialchars($subject_data['name'] ?? 'Subject'); ?>" class="img-fluid">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>
        </main>
    </div>

    <!-- JavaScript -->
    <script>
        // Add any necessary JavaScript here
        document.addEventListener('DOMContentLoaded', function() {
            // Mobile menu toggle
            const hamburger = document.querySelector('.hamburger');
            const sideMenu = document.querySelector('.side-menu');
            const closeMenu = document.querySelector('.close-menu');
            
            if (hamburger && sideMenu) {
                hamburger.addEventListener('click', function() {
                    sideMenu.classList.add('active');
                });
            }
            
            if (closeMenu && sideMenu) {
                closeMenu.addEventListener('click', function() {
                    sideMenu.classList.remove('active');
                });
            }
        });
    </script>
</body>
</html>

