<?php
/**
 * Security Helper Functions
 * 
 * Collection of security-related functions for input validation,
 * sanitization, and protection against common web vulnerabilities.
 * 
 * @author Thamrah Development Team
 * @version 2.0
 * @since 2025-06-14
 */

/**
 * Generate a secure CSRF token
 * 
 * @return string CSRF token
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token']) || 
        !isset($_SESSION['csrf_token_time']) || 
        (time() - $_SESSION['csrf_token_time']) > 1800) { // 30 minutes
        
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        $_SESSION['csrf_token_time'] = time();
    }
    
    return $_SESSION['csrf_token'];
}

/**
 * Verify CSRF token
 * 
 * @param string $token Token to verify
 * @return bool True if valid, false otherwise
 */
function verifyCSRFToken($token) {
    if (!isset($_SESSION['csrf_token']) || 
        !isset($_SESSION['csrf_token_time'])) {
        return false;
    }
    
    // Check if token has expired (30 minutes)
    if ((time() - $_SESSION['csrf_token_time']) > 1800) {
        unset($_SESSION['csrf_token'], $_SESSION['csrf_token_time']);
        return false;
    }
    
    return hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Sanitize input for HTML output
 * 
 * @param string $input Input string
 * @return string Sanitized string
 */
function sanitizeHTML($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES | ENT_HTML5, 'UTF-8');
}

/**
 * Sanitize input for database queries (additional layer)
 * 
 * @param string $input Input string
 * @return string Sanitized string
 */
function sanitizeDB($input) {
    return trim(strip_tags($input));
}

/**
 * Validate email address
 * 
 * @param string $email Email to validate
 * @return string|false Valid email or false if invalid
 */
function validateEmail($email) {
    $email = filter_var(trim($email), FILTER_VALIDATE_EMAIL);
    return $email !== false ? $email : false;
}

/**
 * Validate URL
 * 
 * @param string $url URL to validate
 * @return string|false Valid URL or false if invalid
 */
function validateURL($url) {
    $url = filter_var(trim($url), FILTER_VALIDATE_URL);
    return $url !== false ? $url : false;
}

/**
 * Validate phone number (basic validation)
 * 
 * @param string $phone Phone number to validate
 * @return string|false Valid phone or false if invalid
 */
function validatePhone($phone) {
    $phone = preg_replace('/[^0-9+\-\s\(\)]/', '', $phone);
    return strlen($phone) >= 8 ? $phone : false;
}

/**
 * Generate secure password hash
 * 
 * @param string $password Plain text password
 * @return string Hashed password
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_ARGON2ID, [
        'memory_cost' => 65536, // 64 MB
        'time_cost' => 4,       // 4 iterations
        'threads' => 3          // 3 threads
    ]);
}

/**
 * Verify password against hash
 * 
 * @param string $password Plain text password
 * @param string $hash Stored password hash
 * @return bool True if password matches, false otherwise
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Generate secure random string
 * 
 * @param int $length Length of string to generate
 * @return string Random string
 */
function generateRandomString($length = 32) {
    return bin2hex(random_bytes($length / 2));
}

/**
 * Rate limiting check
 * 
 * @param string $identifier Unique identifier (IP, user ID, etc.)
 * @param int $limit Maximum attempts
 * @param int $window Time window in seconds
 * @return bool True if within limit, false if exceeded
 */
function checkRateLimit($identifier, $limit = 10, $window = 3600) {
    $key = "rate_limit_" . md5($identifier);
    
    if (!isset($_SESSION[$key])) {
        $_SESSION[$key] = ['count' => 0, 'start' => time()];
    }
    
    $data = $_SESSION[$key];
    
    // Reset if window has passed
    if ((time() - $data['start']) > $window) {
        $_SESSION[$key] = ['count' => 1, 'start' => time()];
        return true;
    }
    
    // Check if limit exceeded
    if ($data['count'] >= $limit) {
        return false;
    }
    
    // Increment counter
    $_SESSION[$key]['count']++;
    return true;
}

/**
 * Log security events
 * 
 * @param string $event Event description
 * @param array $context Additional context
 */
function logSecurityEvent($event, $context = []) {
    $log_entry = [
        'timestamp' => date('Y-m-d H:i:s'),
        'event' => $event,
        'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
        'context' => $context
    ];
    
    error_log("SECURITY: " . json_encode($log_entry));
}

/**
 * Check if request is from allowed origin
 * 
 * @param array $allowed_origins List of allowed origins
 * @return bool True if allowed, false otherwise
 */
function checkOrigin($allowed_origins = ['*']) {
    if (in_array('*', $allowed_origins)) {
        return true;
    }
    
    $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
    return in_array($origin, $allowed_origins);
}

/**
 * Validate file upload
 * 
 * @param array $file $_FILES array element
 * @param array $allowed_types Allowed MIME types
 * @param int $max_size Maximum file size in bytes
 * @return array Validation result
 */
function validateFileUpload($file, $allowed_types = [], $max_size = 10485760) {
    $result = ['valid' => false, 'error' => ''];
    
    // Check if file was uploaded
    if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
        $result['error'] = 'No file uploaded or upload error';
        return $result;
    }
    
    // Check file size
    if ($file['size'] > $max_size) {
        $result['error'] = 'File size exceeds maximum allowed size';
        return $result;
    }
    
    // Check MIME type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);
    
    if (!empty($allowed_types) && !in_array($mime_type, $allowed_types)) {
        $result['error'] = 'File type not allowed';
        return $result;
    }
    
    $result['valid'] = true;
    $result['mime_type'] = $mime_type;
    return $result;
}

?>

