<?php
/**
 * Universities API Endpoint
 * 
 * This endpoint retrieves universities based on country ID and locale.
 * Improved with security, performance, and proper error handling.
 * 
 * @author Thamrah Development Team
 * @version 2.0
 * @since 2025-06-14
 */

require_once("db/connection.php");

/**
 * Set secure response headers for API endpoint
 */
function setApiHeaders() {
    header('Content-Type: application/json; charset=utf-8');
    header('Cache-Control: max-age=3600, public');
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
    
    // Handle preflight requests
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(200);
        exit();
    }
}

/**
 * Validate input parameters
 * 
 * @param array $params Input parameters
 * @return array Validation result with status and data/errors
 */
function validateInput($params) {
    $errors = [];
    
    // Validate country_id
    if (!isset($params['country_id']) || empty($params['country_id'])) {
        $errors[] = 'Country ID is required';
    } else {
        $country_id = validateInteger($params['country_id']);
        if ($country_id === false || $country_id <= 0) {
            $errors[] = 'Invalid country ID format';
        }
    }
    
    // Validate locale
    if (!isset($params['locale']) || empty($params['locale'])) {
        $errors[] = 'Locale is required';
    } else {
        $locale = sanitizeInput($params['locale']);
        if (!in_array($locale, ['en', 'ar'])) {
            $errors[] = 'Invalid locale. Must be "en" or "ar"';
        }
    }
    
    if (!empty($errors)) {
        return ['status' => 'error', 'errors' => $errors];
    }
    
    return [
        'status' => 'success',
        'data' => [
            'country_id' => $country_id,
            'locale' => $locale
        ]
    ];
}

/**
 * Get universities by country ID and locale
 * 
 * @param mysqli $connection Database connection
 * @param int $country_id Country ID
 * @param string $locale Language locale
 * @return array Universities data
 */
function getUniversitiesByCountry($connection, $country_id, $locale) {
    try {
        // Use prepared statement to prevent SQL injection
        $query = "SELECT u.id, u.status, ut.name, ut.description 
                  FROM universities u 
                  LEFT JOIN university_translations ut ON u.id = ut.university_id 
                  WHERE u.status = ? AND u.country_id = ? AND ut.locale = ?
                  ORDER BY ut.name ASC";
        
        $result = executeQuery($connection, $query, ['active', $country_id, $locale], 'sis');
        
        if ($result === false) {
            return ['error' => 'Database query failed'];
        }
        
        $universities = [];
        while ($row = $result->fetch_assoc()) {
            $universities[] = [
                'id' => (int)$row['id'],
                'name' => $row['name'] ?? '',
                'description' => $row['description'] ?? '',
                'status' => $row['status']
            ];
        }
        
        return $universities;
        
    } catch (Exception $e) {
        error_log("Error fetching universities: " . $e->getMessage());
        return ['error' => 'Internal server error'];
    }
}

/**
 * Send JSON response with proper HTTP status code
 * 
 * @param array $data Response data
 * @param int $status_code HTTP status code
 */
function sendJsonResponse($data, $status_code = 200) {
    http_response_code($status_code);
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit();
}

// Main execution
try {
    // Set headers
    setApiHeaders();
    
    // Only allow GET requests
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        sendJsonResponse(['error' => 'Method not allowed'], 405);
    }
    
    // Validate input
    $validation = validateInput($_GET);
    if ($validation['status'] === 'error') {
        sendJsonResponse(['error' => 'Validation failed', 'details' => $validation['errors']], 400);
    }
    
    $country_id = $validation['data']['country_id'];
    $locale = $validation['data']['locale'];
    
    // Check database connection
    if (!$con) {
        sendJsonResponse(['error' => 'Database connection failed'], 500);
    }
    
    // Get universities
    $universities = getUniversitiesByCountry($con, $country_id, $locale);
    
    // Check for errors
    if (isset($universities['error'])) {
        sendJsonResponse($universities, 500);
    }
    
    // Return successful response
    sendJsonResponse([
        'status' => 'success',
        'data' => $universities,
        'count' => count($universities),
        'country_id' => $country_id,
        'locale' => $locale
    ]);
    
} catch (Exception $e) {
    error_log("Unexpected error in universities API: " . $e->getMessage());
    sendJsonResponse(['error' => 'Internal server error'], 500);
}

?>

