<?php
/**
 * Database Configuration and Connection Handler
 * 
 * This file provides a secure database connection with proper error handling
 * and configuration management for the Thamrah web application.
 * 
 * @author Thamrah Development Team
 * @version 2.0
 * @since 2025-06-14
 */

// Database configuration - These should be moved to environment variables in production
$config = [
    'host' => 'localhost',
    'username' => 'root', 
    'password' => '',
    'database' => 'thamrah_db',
    'charset' => 'utf8mb4',
    'port' => 3306
];

/**
 * Create a secure database connection with error handling
 * 
 * @return mysqli|false Database connection object or false on failure
 */
function createDatabaseConnection($config) {
    try {
        // Enable error reporting for mysqli
        mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
        
        // Create connection with proper charset
        $connection = new mysqli(
            $config['host'],
            $config['username'], 
            $config['password'],
            $config['database'],
            $config['port']
        );
        
        // Set charset to prevent character encoding issues
        $connection->set_charset($config['charset']);
        
        return $connection;
        
    } catch (mysqli_sql_exception $e) {
        // Log error securely (don't expose database details to users)
        error_log("Database connection failed: " . $e->getMessage());
        
        // Return false to indicate connection failure
        return false;
    }
}

/**
 * Execute a prepared statement safely
 * 
 * @param mysqli $connection Database connection
 * @param string $query SQL query with placeholders
 * @param array $params Parameters to bind
 * @param string $types Parameter types (s=string, i=integer, d=double, b=blob)
 * @return mysqli_result|false Query result or false on failure
 */
function executeQuery($connection, $query, $params = [], $types = '') {
    try {
        $stmt = $connection->prepare($query);
        
        if (!$stmt) {
            error_log("Prepare failed: " . $connection->error);
            return false;
        }
        
        // Bind parameters if provided
        if (!empty($params) && !empty($types)) {
            $stmt->bind_param($types, ...$params);
        }
        
        $stmt->execute();
        $result = $stmt->get_result();
        $stmt->close();
        
        return $result;
        
    } catch (mysqli_sql_exception $e) {
        error_log("Query execution failed: " . $e->getMessage());
        return false;
    }
}

/**
 * Sanitize input to prevent XSS attacks
 * 
 * @param string $input User input
 * @return string Sanitized input
 */
function sanitizeInput($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Validate and sanitize integer input
 * 
 * @param mixed $input User input
 * @return int|false Validated integer or false if invalid
 */
function validateInteger($input) {
    $filtered = filter_var($input, FILTER_VALIDATE_INT);
    return $filtered !== false ? $filtered : false;
}

// Create the database connection
$con = createDatabaseConnection($config);

// Check if connection was successful
if (!$con) {
    // In production, redirect to an error page instead of dying
    die("Database connection failed. Please try again later.");
}

// Set timezone for consistent date/time handling
date_default_timezone_set('Asia/Kuwait');

?>

