<?php
/**
 * Configuration Management Class
 * 
 * Centralized configuration management for the Thamrah application
 * with environment-specific settings and security best practices.
 * 
 * @author Thamrah Development Team
 * @version 2.0
 * @since 2025-06-14
 */

class Config {
    
    /**
     * Application environment (development, staging, production)
     */
    const ENVIRONMENT = 'development';
    
    /**
     * Application version
     */
    const VERSION = '2.0.0';
    
    /**
     * Database configuration
     * In production, these should be loaded from environment variables
     */
    const DB_CONFIG = [
        'host' => 'localhost',
        'username' => 'root',
        'password' => '',
        'database' => 'thamrah_db',
        'charset' => 'utf8mb4',
        'port' => 3306
    ];
    
    /**
     * Security configuration
     */
    const SECURITY = [
        'session_timeout' => 3600, // 1 hour
        'max_login_attempts' => 5,
        'password_min_length' => 8,
        'csrf_token_lifetime' => 1800, // 30 minutes
        'allowed_file_types' => ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx'],
        'max_file_size' => 10485760 // 10MB
    ];
    
    /**
     * Application settings
     */
    const APP = [
        'name' => 'Thamrah',
        'timezone' => 'Asia/Kuwait',
        'default_locale' => 'ar',
        'supported_locales' => ['ar', 'en'],
        'items_per_page' => 20,
        'cache_lifetime' => 3600
    ];
    
    /**
     * API configuration
     */
    const API = [
        'rate_limit' => 100, // requests per hour
        'timeout' => 30, // seconds
        'allowed_origins' => ['*'], // In production, specify actual domains
        'version' => 'v1'
    ];
    
    /**
     * Email configuration
     */
    const EMAIL = [
        'smtp_host' => 'localhost',
        'smtp_port' => 587,
        'smtp_username' => '',
        'smtp_password' => '',
        'from_email' => 'noreply@thamrah.com',
        'from_name' => 'Thamrah'
    ];
    
    /**
     * File upload configuration
     */
    const UPLOAD = [
        'path' => '/uploads/',
        'temp_path' => '/tmp/',
        'max_size' => 10485760, // 10MB
        'allowed_types' => ['image/jpeg', 'image/png', 'image/gif', 'application/pdf']
    ];
    
    /**
     * Get configuration value by key
     * 
     * @param string $section Configuration section
     * @param string $key Configuration key
     * @param mixed $default Default value if key not found
     * @return mixed Configuration value
     */
    public static function get($section, $key = null, $default = null) {
        $config = constant("self::{$section}");
        
        if ($key === null) {
            return $config;
        }
        
        return $config[$key] ?? $default;
    }
    
    /**
     * Check if we're in development environment
     * 
     * @return bool
     */
    public static function isDevelopment() {
        return self::ENVIRONMENT === 'development';
    }
    
    /**
     * Check if we're in production environment
     * 
     * @return bool
     */
    public static function isProduction() {
        return self::ENVIRONMENT === 'production';
    }
    
    /**
     * Get database DSN string
     * 
     * @return string
     */
    public static function getDatabaseDSN() {
        $config = self::DB_CONFIG;
        return "mysql:host={$config['host']};port={$config['port']};dbname={$config['database']};charset={$config['charset']}";
    }
}

?>

